#ifndef SAMPLING_H
#define SAMPLING_H

#include <vector>
#include <random>
#include <memory>
#include <map>
#include <set>
#include <iostream>
#include "Helper.hpp"
#include "Randomness.hpp"

/**
 * @brief Sampling operations
 */
class Sampling
{
public:
	
	/**
	 * @brief Sample without replacement
	 * 
	 * Requires InputIterator
	 */
	template<typename InputIterator, typename ResultType>
	static std::unique_ptr<std::vector<ResultType>> sampleWithoutReplacement(InputIterator first, InputIterator last, size_t sizeOfSample);
	
	/**
	 * @brief Sample without replacement
	 * 
	 * Requires RandomAccessIterator
	 */
	template<typename RandomAccessIterator, typename ResultType>
	static std::unique_ptr<std::vector<ResultType>> sampleWithoutReplacementFast(RandomAccessIterator first, RandomAccessIterator last, size_t sizeOfSample);

	/**
	 * @brief Sample with replacement
	 * 
	 * Output order is not random!
	 */
	template<typename InputIterator, typename ResultType>
	static std::unique_ptr<std::vector<ResultType>> sampleWithReplacement(InputIterator first, InputIterator last, std::vector<double> & weights, size_t sizeOfSample);

	/**
	 * @brief Sample with replacement
	 * 
	 * Output order is not random!
	 */
	template<typename InputIterator, typename ResultType>
	static std::unique_ptr<std::vector<ResultType>> sampleWithReplacement(InputIterator first, InputIterator last, size_t sizeOfSet, size_t sizeOfSample);
};

template<typename InputIterator, typename ResultType>
std::unique_ptr<std::vector<ResultType>> Sampling::sampleWithoutReplacement(InputIterator first, InputIterator last, size_t sizeOfSample)
{
	std::unique_ptr<std::vector<ResultType>> buckets(new std::vector<ResultType>(sizeOfSample));
	std::mt19937 * gen = Randomness::getMT19937();
	std::uniform_real_distribution<double> sampleDice(0, 1);
	std::uniform_int_distribution<size_t> bucketDice(0, sizeOfSample-1);

	// Take first (sizeOfSample) elements as initial sample
	size_t numberOfElements = 0;
	for(auto it = buckets->begin(); it != buckets->end() && first != last; ++it)
	{
		*it = *(first++);
		++numberOfElements;
	}

	// Less elements than buckets?
	if(sizeOfSample > numberOfElements)
		return std::unique_ptr<std::vector<ResultType>>(new std::vector<ResultType>(buckets->begin(), buckets->begin()+numberOfElements));

	// Sample from remaining (n+1)...(last) elements
	size_t index = sizeOfSample+1;
	while(first != last)
	{
		if(sampleDice(*gen) < double(sizeOfSample)/double(index))
			(*buckets)[bucketDice(*gen)] = *first;
		++first;
		++index;
	}

	return buckets;
}

template<typename RandomAccessIterator, typename ResultType>
std::unique_ptr<std::vector<ResultType>> Sampling::sampleWithoutReplacementFast(RandomAccessIterator first, RandomAccessIterator last, size_t sizeOfSample)
{
	unsigned long long n = last-first;

	if(n <= sizeOfSample)
		return std::unique_ptr<std::vector<ResultType>>(new std::vector<ResultType>(first, last));

	std::mt19937 * gen = Randomness::getMT19937();
	std::uniform_real_distribution<double> sampleDice(0, n-1);
	std::set<size_t> selected;
	std::vector<size_t> indices;
	indices.reserve(sizeOfSample);
	while(selected.size() < sizeOfSample)
	{
		size_t element = sampleDice(*gen);
		if(selected.count(element) == 0)
		{
			selected.insert(element);
			indices.push_back(element);
		}
	}

	std::unique_ptr<std::vector<ResultType>> buckets(new std::vector<ResultType>());
	buckets->reserve(sizeOfSample);

	for(auto it = indices.begin(); it != indices.end(); ++it)
		buckets->push_back(*(first + *it));

	return buckets;
}

template<typename InputIterator, typename ResultType>
std::unique_ptr<std::vector<ResultType>> Sampling::sampleWithReplacement(InputIterator first, InputIterator last, std::vector<double> & weights, size_t sizeOfSample)
{
	std::mt19937 * gen = Randomness::getMT19937();
	std::discrete_distribution<size_t> d(weights.begin(), weights.end());

	// Sample
	std::multimap<size_t, size_t> samples;
	for(size_t i = 0; i < sizeOfSample; ++i)
		samples.insert(std::pair<size_t,size_t>(d(*gen), i));

	// Construct sample set
	std::unique_ptr<std::vector<ResultType>> buckets(new std::vector<ResultType>(sizeOfSample));
	size_t index = 0;
	for(auto it = first; it != last; ++it)
	{
		auto buck = samples.equal_range(index);
		for(auto itbuck = buck.first; itbuck != buck.second; ++itbuck)
			(*buckets)[itbuck->second] = *it;
		++index;
	}

	return buckets;
}

template<typename InputIterator, typename ResultType>
std::unique_ptr<std::vector<ResultType>> Sampling::sampleWithReplacement(InputIterator first, InputIterator last, size_t sizeOfSet, size_t sizeOfSample)
{
	std::mt19937 * gen = Randomness::getMT19937();
	std::uniform_int_distribution<size_t> d(0, sizeOfSet-1);

	// Sample
	std::multimap<size_t, size_t> samples;
	for(size_t i = 0; i < sizeOfSample; ++i)
		samples.insert(std::pair<size_t,size_t>(d(*gen), i));

	// Construct sample set
	std::unique_ptr<std::vector<ResultType>> buckets(new std::vector<ResultType>(sizeOfSample));
	size_t index = 0;
	for(auto it = first; it != last; ++it)
	{
		auto buck = samples.equal_range(index);
		for(auto itbuck = buck.first; itbuck != buck.second; ++itbuck)
			(*buckets)[itbuck->second] = *it;
		++index;
	}

	return buckets;
}

#endif
